import { Controller, Get, Req, Sse } from '@nestjs/common';
import { MaintenanceService } from './maintenance.service';
import { EvaluationResult, PriorityList } from '../fuzzy-core/fuzzy.types';
import { Observable } from 'rxjs';
import { IncomingMessage } from 'http';
import { map } from 'rxjs/operators';
import {
  ApiOperation,
  ApiProduces,
  ApiResponse,
  ApiTags,
} from '@nestjs/swagger';

/**
 * MaintenanceController handles the maintenance-related endpoints.
 * It provides an endpoint to get the latest evaluation results and
 * a server-sent events (SSE) endpoint to stream evaluation results.
 */
@Controller('maintenance')
@ApiTags('Maintenance Evaluation')
export class MaintenanceController {
  constructor(private maintenanceService: MaintenanceService) {}

  /**
   * Endpoint to get the latest evaluation results.
   * @return {EvaluationResult[]} Array of the latest evaluation results.
   */
  @Get('/latest/evaluation')
  @ApiOperation({ summary: 'Get the latest evaluation results' })
  @ApiProduces('application/json')
  @ApiResponse({
    status: 200,
    description: 'Latest evaluation results',
    isArray: true,
  })
  getLatestEvaluation(): EvaluationResult[] {
    return this.maintenanceService.getLatestEvaluations();
  }

  /**
   * SSE endpoint to stream evaluation results.
   * @description This endpoint streams evaluation results as they become available based on the real-time sensor data.
   * @param request Incoming HTTP request.
   * @return {Observable<{ data: EvaluationResult[] }>} Observable of evaluation results.
   */
  @Sse('evaluate')
  @ApiOperation({ summary: 'Stream evaluation results' })
  @ApiProduces('text/event-stream')
  @ApiResponse({
    status: 200,
    description: 'Stream of evaluation results',
    isArray: true,
  })
  getEvaluations(
    @Req() request: IncomingMessage,
  ): Observable<{ data: PriorityList }> {
    return this.maintenanceService
      .getEvaluations(request)
      .pipe(map((evaluations) => ({ data: { priorityList: evaluations } })));
  }
}
